<?php
/*------------------------------------------------------------------------------
  DashboardData.php 2021-08-01
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2021 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  -----------------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Dashboard\ValueObjects;

use Gambio\Admin\Modules\SetupWizard\Collections\SetupWizardStepCollection;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\PreferredCategory as StatisticsPreferredCategory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\PreferredPeriod as StatisticsPreferredPeriod;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\Result as StatisticsResult;

/**
 * Class DashboardData
 * @package Gambio\Admin\Modules\Dashboard\ValueObjects
 */
class DashboardData implements \JsonSerializable
{
    /**
     * @var string
     */
    private $shopKey;
    /**
     * @var string
     */
    private $shopUrl;
    
    /**
     * @var string
     */
    private $shopVersion;
    
    /**
     * @var StatisticsResult
     */
    protected $statistics;
    
    /**
     * @var SetupWizardStepCollection
     */
    protected $steps;
    
    /**
     * @var bool
     */
    private $isExternalSocialMediaEmbedsAllowed;
    
    /**
     * @var StatisticsPreferredCategory
     */
    private $userPreferredCategory;
    
    /**
     * @var StatisticsPreferredPeriod
     */
    private $userPreferredPeriod;
    
    /**
     * @var ShopOrigin
     */
    private $shopOrigin;
    
    
    /**
     * DashboardData constructor.
     *
     * @param string                      $shopKey
     * @param string                      $shopUrl
     * @param string                      $shopVersion
     * @param StatisticsResult            $statistics
     * @param SetupWizardStepCollection   $steps
     * @param bool                        $isExternalSocialMediaEmbedsAllowed
     * @param StatisticsPreferredCategory $userPreferredCategory
     * @param StatisticsPreferredPeriod   $userPreferredPeriod
     * @param ShopOrigin                  $shopOrigin
     */
    public function __construct(
        string $shopKey,
        string $shopUrl,
        string $shopVersion,
        StatisticsResult $statistics,
        SetupWizardStepCollection $steps,
        bool $isExternalSocialMediaEmbedsAllowed,
        StatisticsPreferredCategory $userPreferredCategory,
        StatisticsPreferredPeriod $userPreferredPeriod,
        ShopOrigin $shopOrigin
    ) {
        $this->statistics                         = $statistics;
        $this->steps                              = $steps;
        $this->shopKey                            = $shopKey;
        $this->shopUrl                            = $shopUrl;
        $this->shopVersion                        = $shopVersion;
        $this->isExternalSocialMediaEmbedsAllowed = $isExternalSocialMediaEmbedsAllowed;
        $this->userPreferredCategory              = $userPreferredCategory;
        $this->userPreferredPeriod                = $userPreferredPeriod;
        $this->shopOrigin                         = $shopOrigin;
    }
    
    
    /**
     * @return StatisticsPreferredCategory
     */
    public function userPreferredCategory(): StatisticsPreferredCategory
    {
        return $this->userPreferredCategory;
    }
    
    
    /**
     * @return StatisticsPreferredPeriod
     */
    public function userPreferredPeriod(): StatisticsPreferredPeriod
    {
        return $this->userPreferredPeriod;
    }
    
    
    /**
     * @return string
     */
    public function shopKey(): string
    {
        return $this->shopKey;
    }
    
    
    /**
     * @return string
     */
    public function shopUrl(): string
    {
        return $this->shopUrl;
    }
    
    
    /**
     * @return string
     */
    public function shopVersion(): string
    {
        return $this->shopVersion;
    }
    
    
    /**
     * @return StatisticsResult
     */
    public function statistics(): StatisticsResult
    {
        return $this->statistics;
    }
    
    
    /**
     * @return ShopOrigin
     */
    public function shopOrigin(): ShopOrigin
    {
        return $this->shopOrigin;
    }
    
    
    public function jsonSerialize()
    {
        return [
            'shopKey'               => $this->shopKey(),
            'shopUrl'               => $this->shopUrl(),
            'shopVersion'           => $this->shopVersion(),
            'categories'            => $this->statistics->categories(),
            'userPreferredCategory' => $this->userPreferredCategory->jsonSerialize(),
            'userPreferredPeriod'   => $this->userPreferredPeriod->jsonSerialize(),
            'statistics'            => $this->statistics->statistics(),
            'setupData'             => $this->steps,
            'loadExternalContents'  => $this->isExternalSocialMediaEmbedsAllowed,
            'shopOrigin'            => $this->shopOrigin(),
        ];
    }
}